// Expected global variables:
/*global jQuery toolwindowHelpers */

(function ($) {
    /// <summary>
    ///     HtmlTreeView - JQuery extension for constructing a tree view consisting of tags and attributes, 
    ///     The html tree view object can be initialized by calling htmlTreeView on a jQuery object.
    ///     Requires:
    ///     - htmltree.css
    ///     - itemCollapsedIcon.png
    ///     - itemExpandedIcon.png
    ///     - itemCollapsedDarkThemeIcon.png
    ///     - itemExpandedDarkThemeIcon.png
    /// </summary>
    
    // The number of elements to display before the 'show all' link is shown
    var initialElementLimit = 200;
    
    var methods = {

        // Default constructor
        init: function () {
            /// <summary>
            ///     Creates a new HtmlTreeView for a BPT-HtmlTree-Container
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object for the new HtmlTreeView
            /// </returns>
            
            var rootElement = $("<div class='BPT-HtmlTree'></div>");
            this.append(rootElement);
            
            var useDarkTheme = toolwindowHelpers.isDarkThemeBackground(rootElement);
            if (useDarkTheme) {
                rootElement.addClass("BPT-Tree-DarkTheme");
            } else {
                rootElement.removeClass("BPT-Tree-DarkTheme");
            }
            
            // Attach the event handlers if we need to
            if (!rootElement.data("attachedHandlers")) {
                var container = rootElement.parent(".BPT-HtmlTree-Container");
                
                container.bind("mousedown.htmlTreeView", function (event) {
                    $(this).data("mouseActivate", true);
                });
                
                container.bind("click.htmlTreeView", function (event) {
                    var element = $(event.target);
                    if (!element.is(".BPT-HtmlTree-ChildCollection")) {
                        var row = element.closest(".BPT-HtmlTreeItem");
                        
                        if (row.length > 0) {
                            // If they clicked the expand icon, toggle the row
                            if (element.hasClass("BPT-HtmlTreeItem-ExpandIcon")) {
                                methods.toggle.call(row);
                            }
                            methods.select.call(row);
                        }
                    }
                });
                
                container.bind("dblclick.htmlTreeView", function (event) {
                    var element = $(event.target);
                    var item = element.closest(".BPT-HtmlTreeItem, .BPT-HTML-Attribute-Section, .BPT-HTML-Text, .BPT-HtmlTree-ChildCollection-Pager");
                    
                    if (item.length > 0) {
                        if (item.hasClass("BPT-HtmlTreeItem")) {
                            // Double clicking the row will expand/collapse it
                            if (item.hasClass("BPT-HtmlTreeItem-Collapsed") || item.hasClass("BPT-HtmlTreeItem-Expanded")) {
                                if (!element.hasClass("BPT-HtmlTreeItem-ExpandIcon")) {
                                    methods.toggle.call(item);
                                }
                            }
                        } else if (item.hasClass("BPT-HTML-Attribute-Section") || item.hasClass("BPT-HTML-Text")) {
                            // Double clicking an attribute or inline text will edit it
                            var row = item.parents(".BPT-HtmlTreeItem:first");
                            if (row.length > 0) {
                                var editCallback = row.data("editCallback");
                                editCallback = (editCallback ? editCallback : row.parent().data("editCallback"));
                                
                                if (editCallback) {
                                    editCallback(row, item);
                                    event.stopPropagation();
                                }
                            }
                        }
                    }             
                });                
                
                container.bind("focus.htmlTreeView", function (event) {
                    if (!$(this).data("mouseActivate")) {
                        var element = $(this);
                        var selected = element.children(".BPT-HtmlTree").htmlTreeView("getSelected");
                        if (selected.length === 0) {
                            selected = element.find(".BPT-HtmlTreeItem:not(.BPT-HtmlTreeItem-HiddenRoot):first").htmlTreeView("select");
                        }
                        
                        if (selected && selected.length > 0) {
                            var wasScrolled = toolwindowHelpers.scrollIntoView(selected.children(".BPT-HtmlTreeItem-Header")[0], selected.closest(".BPT-HtmlTree-ScrollContainer")[0]);
                            if (wasScrolled) {
                                event.preventDefault();
                                return false;
                            }
                        }
                    }
                    // Always prevent the default scrolling behavior on focus as it jumps to the top
                    event.preventDefault();
                    $(this).data("mouseActivate", false);
                });
                
                container.bind("keydown.htmlTreeView", function (event) {
                    if (event.keyCode >= 37 && event.keyCode <= 40) { // Arrow Keys
                    
                        // Don't do anything if we are inside an input
                        if ($(document.activeElement).is(":input")) {
                            return;
                        }
                        
                        var selected = methods.getSelected.call($(this).children(":first"));
                        
                        var moveUp = function (toParent) {
                            /// <summary>
                            ///     Moves the selection up to the previous node
                            /// </summary>
                            /// <param name="toParent" type="Boolean" optional="true">
                            ///     Optional parameter specifying if the jumps to parent nodes are allowed (default false)
                            /// </param>  
                            
                            var newElement = null;
                            var sibling = selected.prev(".BPT-HtmlTreeItem:not(.BPT-HtmlTreeItem-HiddenRoot):last");
                            if (sibling.length > 0 && !toParent) {
                                // Find the last child
                                newElement = sibling.find(".BPT-HtmlTreeItem:not(.BPT-HtmlTreeItem-HiddenRoot):last");
                                
                                if (newElement.length === 0) {
                                    // Use the sibling instead
                                    newElement = sibling;
                                }
                            } else {
                                newElement = selected.parents(".BPT-HtmlTreeItem:not(.BPT-HtmlTreeItem-HiddenRoot):first");
                            }
                            
                            if (newElement && newElement.length > 0) {
                                methods.select.call(newElement);
                                toolwindowHelpers.scrollIntoView(newElement.children(".BPT-HtmlTreeItem-Header")[0], newElement.closest(".BPT-HtmlTree-ScrollContainer")[0]);
                                event.preventDefault();
                                return false;
                            }
                        };
                        
                        var moveDown = function () {
                            /// <summary>
                            ///     Moves the selection down to the next node
                            /// </summary>
                            
                            var newElement = selected.find(".BPT-HtmlTreeItem:not(.BPT-HtmlTreeItem-HiddenRoot):first");
                            newElement = (newElement.length > 0 ? newElement : selected.next(".BPT-HtmlTreeItem:first"));
                            
                            var searchedParent = selected;
                            while (newElement.length === 0) {
                                searchedParent = searchedParent.parents(".BPT-HtmlTreeItem:not(.BPT-HtmlTreeItem-HiddenRoot)");
                                if (searchedParent.length === 0) {
                                    break;
                                }
                                newElement = searchedParent.next(".BPT-HtmlTreeItem:not(.BPT-HtmlTreeItem-HiddenRoot):first");
                            }
                            
                            if (newElement && newElement.length > 0) {
                                methods.select.call(newElement);
                                toolwindowHelpers.scrollIntoView(newElement.children(".BPT-HtmlTreeItem-Header")[0], newElement.closest(".BPT-HtmlTree-ScrollContainer")[0]);
                                event.preventDefault();
                                return false;
                            }
                        };
                        
                        if (selected.length > 0) {
                            switch (event.keyCode) {
                                case 37: // Left(37)
                                    if (selected.hasClass("BPT-HtmlTreeItem-Expanded")) {
                                        methods.toggle.call(selected);
                                    } else {
                                        moveUp(true);
                                    }
                                    break;
                                    
                                case 38: // Up(38)
                                    moveUp();
                                    break;
                                    
                                case 39: // Right(39)
                                    if (selected.hasClass("BPT-HtmlTreeItem-Collapsed")) {
                                        methods.toggle.call(selected);
                                    } else if (selected.hasClass("BPT-HtmlTreeItem-Expanded")) {    
                                        moveDown();
                                    }  
                                    break;
                                    
                                case 40: // Down(40)
                                    moveDown();
                                    break;
                            }                        
                        }
                        
                        // Prevent the tree from scrolling with the arrows (matches solution explorer behavior)
                        event.preventDefault();
                        return false;  
                        
                    } else if (event.keyCode === 13 || event.keyCode === 32) { // Enter(13) or Space(32) 
                    
                        // Ensure we are not currently editing
                        if (document.activeElement && document.activeElement.type !== "text") {
                        
                            if (event.keyCode === 32) { // Space(32)
                                // We didn't have an item to click and they were pressing space,
                                // so stop the scroll container from scrolling.
                                return false;
                            }
                            
                            // Find out if this is an editable text node
                            var selectedNode = methods.getSelected.call($(this).children(":first"));
                            
                            // Find out if this is a 'show all' link
                            var isShowAllLink = selectedNode.hasClass("BPT-HtmlTree-ChildCollection-ShowAll");
                            if (isShowAllLink) {
                                selectedNode.click();
                                return;
                            }
                            
                            // Find out if this is an editable text node
                            var text = selectedNode.find(".BPT-HtmlTreeItem-Header:first > .BPT-HTML > .BPT-HTML-Text:last");
                            if (text.length === 1) {
                                // Found a text node, so emulate a double click
                                event.preventDefault();
                                event.stopImmediatePropagation();
                                text.trigger("dblclick");
                            }
                        }
                    }
                });
                container = null;
                rootElement.data("attachedHandlers", true);
            }
            
            return this;
        },
        
        destroy : function () {
            /// <summary>
            ///     Disposes of an HtmlTreeView and removes all data and event handlers
            /// </summary>  
            
            // Remove event handlers
            if (this.data("attachedHandlers")) {
                this.children(".BPT-HtmlTree-Container").unbind(".htmlTreeView");
                this.data("attachedHandlers", false);
            }
        },
        
        addRootElement: function (uid, tag, rootTagToShow, toggleCallback, editCallback, selectCallback) {
            /// <summary>
            ///     Adds a single element that will act as the root for this HtmlTreeView
            /// </summary>
            /// <param name="uid" type="String">
            ///     A unique id to assign to this element
            /// </param>  
            /// <param name="tag" type="String">
            ///     The tag to use in the display (#document tags will be hidden)
            /// </param>  
            /// <param name="rootTagToShow" type="String">
            ///     The text to use for the root element's tag.
            ///     If this is null or an empty string, no root node will be shown.
            /// </param>  
            /// <param name="toggleCallback" type="Function">
            ///     An optional callback that will be triggered when the element is toggled
            /// </param>  
            /// <param name="editCallback" type="Function">
            ///     An optional callback that will be triggered when an attribute is double clicked
            /// </param>              
            /// <param name="selectCallback" type="Function">
            ///     An optional callback that will be triggered when the element is selected
            /// </param>            
            /// <returns type="Object">
            ///     The jquery object that was created
            /// </returns> 
            
            if (toggleCallback) {
                this.data({
                    toggleCallback: toggleCallback
                });
            }
            
            var newElements = [{uid: uid, tag: tag, text: "", hasChildren: (toggleCallback ? true : false), attributes: null, rootTagToShow: rootTagToShow}];
            
            var rootElement = methods.addElements.call(this, newElements, toggleCallback, editCallback, selectCallback).children(":first");
            return rootElement;
        },
        
        addElements: function (elements, toggleCallback, editCallback, selectCallback, keepExistingElements, stopAutoScroll) {
            /// <summary>
            ///     Adds an array of elements to the HtmlTreeItem
            /// </summary>
            /// <param name="elements" type="Array">
            ///     An array of objects that describe the children elements, in the following format:
            ///     [{uid: String, tag: String, text: String, hasChildren: Boolean, attributes: [{name:String, value: String}]}]
            /// </param>  
            /// <param name="toggleCallback" type="Function">
            ///     An optional callback that will be triggered when a child element is toggled
            /// </param>  
            /// <param name="editCallback" type="Function">
            ///     An optional callback that will be triggered when an attribute is double clicked
            /// </param>              
            /// <param name="selectCallback" type="Function">
            ///     An optional callback that will be triggered when a child element is selected
            /// </param>   
            /// <param name="keepExistingElements" type="Boolean" optional="true">
            ///     An optional parameter that specifies if we should keep any existing children
            ///     If false (or not specified) all children elements will be replaced with the new ones
            /// </param> 
            /// <param name="stopAutoScroll" type="Boolean" optional="true">
            ///     An optional parameter that specifies if we should not auto scroll the new items into view
            /// </param> 
            /// <returns type="Object">
            ///     A jquery object that was created that contains all the new elements wrapped in a span
            /// </returns> 
            
            var newElementsString = "<span class='BPT-HtmlTree-ChildCollection'>";
            
            var isShowingAll = true;
            var elementCount = elements.length;
            
            if (!this.data("forceShowAll")) {
                // Set the limit of nodes to display
                if (elementCount > initialElementLimit) {
                    elementCount = initialElementLimit;
                    isShowingAll = false;
                }
            }
            
            var existingIdMap = {};
            if (keepExistingElements) {
                // Generate a map of the existing element id's, so we don't overwrite ones that haven't changed
                var existingElements = this.children(".BPT-HtmlTree-ChildCollection").children();
                for (var elementIndex = 0; elementIndex < existingElements.length; elementIndex++) {
                    var uid = $(existingElements[elementIndex]).attr("data-id");
                    existingIdMap[uid] = true;
                }
            }
            
            for (var i = 0; i < elementCount; i++) {
            
                // Get the properties of this new element
                var id = elements[i].uid;
                var tag = elements[i].tag;
                var text = elements[i].text;
                var isExpandable = elements[i].hasChildren;
                var attributes = elements[i].attributes;
                var rootTagToShow = elements[i].rootTagToShow;
                
                if (!tag && (!text || !$.trim(text))) {
                    // Ignore empty text nodes
                    continue;
                }
                
                if (keepExistingElements && existingIdMap[id]) {
                    // Add a fake element as this id already exists and so we will just replace it with the original one
                    newElementsString += "<div class='replaceMe' data-id='" + id + "'></div>";
                    continue;
                }
                
                if (text) {
                    // Escape the < > for the text
                    text = toolwindowHelpers.htmlEscape(text);
                }
                
                // Create the header and footer
                var header;
                var footer;
                if (tag === "#document") {
                    // Document nodes
                    var rootHeader = "";
                    var rootFooter = "";
                    if (rootTagToShow) {
                        // Show a tag name for this document
                        var safeRootTag = toolwindowHelpers.htmlEscape(rootTagToShow);
                        rootHeader = "<span class='BPT-HTML'>&lt;</span><span class='BPT-HTML-Tag'>" + safeRootTag + "</span><span class='BPT-HTML'>&gt;</span>";
                        rootFooter = "<span class='BPT-HTML'>&lt;/</span><span class='BPT-HTML-Tag'>" + safeRootTag + "</span><span class='BPT-HTML'>&gt;</span>";
                    }
                    header = "<span class='BPT-HTML-Document'>" + rootHeader + "</span>";
                    footer = "<span class='BPT-HTML-Document'>" + rootFooter + "</span>";
                } else if (tag === "#doctype") {
                    // DocType nodes
                    header = "<span class='BPT-HTML-DocType'></span>";
                    footer = "<span class='BPT-HTML-DocType'></span>";
                } else if (tag === "#comment") {
                    // Comment nodes
                    header = "<span class='BPT-HTML-Comment'>&lt;!--</span>";
                    footer = "<span class='BPT-HTML-Comment'>--&gt;</span>";
                } else if (tag === null || tag === undefined) {
                    // Text nodes
                    header = "<span class='BPT-HTML-Text'></span>";
                    footer = "<span class='BPT-HTML-Text'></span>";
                } else {
                    // For All other nodes - Create the attributes if we have any
                    var attributesString = "";
                    if (attributes && attributes.length > 0) {
                        for (var j = 0; j < attributes.length; j++) {
                            attributesString += "<span class='BPT-HTML-Attribute-Section'> <span class='BPT-HTML-Attribute'>" + attributes[j].name + "</span><span class='BPT-HTML-Operator'>=</span>\"" +
                                                "<span class='BPT-HTML-Value' data-attrName='" + attributes[j].name + "'>" + toolwindowHelpers.htmlEscape(attributes[j].value) + "</span>\"</span>";
                        }
                    }
                    
                    header = "&lt;<span class='BPT-HTML-Tag'>" + tag + "</span>" + attributesString + "<span class='BPT-HTML'>&gt;</span>";
                    footer = "&lt;/<span class='BPT-HTML-Tag'>" + tag + "</span><span class='BPT-HTML'>&gt;</span>";
                }

                var textContent = (text ? "<span class='BPT-HTML-Text'>" + text + "</span>" : "");
                
                var collapsedFooter = "<span class='BPT-HtmlTreeItem-CollapsedFooter'>" + (isExpandable ? "..." : "") + "<span class='BPT-HTML'>" + footer + "</span></span>";
                var expandIcon = (isExpandable ? "<div class='BPT-HtmlTreeItem-ExpandIcon' />" : "");
                var expandClass = (isExpandable ? " BPT-HtmlTreeItem-Collapsed" : "");
                
                // Document nodes should be hidden
                if (tag === "#document" && !rootTagToShow) {
                    textContent = "";
                    collapsedFooter = "";
                    expandIcon = "";
                    expandClass = " BPT-HtmlTreeItem-HiddenRoot BPT-HtmlTreeItem-Collapsed";
                }
                
                // Check for node numbering
                var nodeIndex = "";
                if (this.attr("data-tag") === "NodeList" || this.attr("data-tag") === "HtmlCollection") {
                    nodeIndex = "<span class='BPT-HTML BPT-HTML-Text BPT-HTML-Numbering'>" + i + "</span>";
                }
                
                newElementsString += "<div class='BPT-HtmlTreeItem" + expandClass + "' data-id='" + id + "' data-tag='" + (tag ? tag : "") + "'>" + expandIcon + "<div class='BPT-HtmlTreeItem-Header'>" + nodeIndex + "<span class='BPT-HTML'>" + header + textContent + "</span>" + collapsedFooter + "</div><div class='BPT-HtmlTreeItem-Footer'><span class='BPT-HTML'>" + footer + "</span></div></div>";
            }

            if (!isShowingAll) {
                // Add the 'show all' link
                newElementsString += "<span class='BPT-HtmlTree-ChildCollection-ShowAll BPT-HtmlTreeItem'>" + "Showing " + elementCount + " of " + elements.length + ". Click to show all" + "</span>";
            }
            
            newElementsString += "</span>";

            // Create the new children
            var childrenCollection = $(newElementsString);
            if (toggleCallback || editCallback || selectCallback) {
                childrenCollection.data({
                    toggleCallback: toggleCallback,
                    editCallback: editCallback,
                    selectCallback: selectCallback
                });
            }
            
            var isFirstRow = !(this.hasClass("BPT-HtmlTreeItem"));
            if (isFirstRow) {
                // Append a new children collection span to the tree
                this.children(".BPT-HtmlTree").append(childrenCollection);
            } else {
                // Check to see if we need to replace nodes before appending them
                if (keepExistingElements) {
                    var existingChildrenCollection = this.children(".BPT-HtmlTree-ChildCollection");
                    if (existingChildrenCollection.length > 0) {
                        // Go through the new children and replace any fake nodes
                        var replaceableChildren = childrenCollection.children("div.replaceMe");
                        for (var index = 0; index < replaceableChildren.length; index++) {
                            // Get the id
                            var replaceableChild = $(replaceableChildren[index]);
                            var idToReplace = replaceableChild.attr("data-id");
                            
                            // Replace the node
                            var existingElement = existingChildrenCollection.children("div.BPT-HtmlTreeItem[data-id='" + idToReplace + "']");
                            replaceableChild.replaceWith(existingElement);
                        }
                        
                        // Check if the selected item is being removed, if so select another row
                        var selectedId = methods.getSelected.call(this).attr("data-id");
                        if (selectedId && existingIdMap[selectedId]) {
                            var found = childrenCollection.children("[data-id='" + selectedId + "'");
                            if (found.length === 0) {
                                // Select the closest available row
                                methods.select.call(this.closest(".BPT-HtmlTreeItem"));
                            }
                        }
                        
                        // Remove the existing collection, so we can replace it with the new one
                        existingChildrenCollection.remove();
                    }
                }
                
                // Append a new children collection span after the item's header
                this.children(".BPT-HtmlTreeItem-Header").after(childrenCollection);
            }

            if (!isShowingAll) {
                // We need to add the link handler
                var row = this;
                var showAll = childrenCollection.children(".BPT-HtmlTree-ChildCollection-ShowAll");
                if (showAll.length > 0) {
                
                    // Add the handler for clicking the link
                    showAll.bind("click", function (event) {
                        if (event.type === "click") {
                            row.data("forceShowAll", true);
                            methods.toggle.call(row);
                            methods.toggle.call(row);
                            row = null;
                        }
                    });
                }
            }
            
            if (!stopAutoScroll) {
                // Scroll the element to the top if the last child is out of view
                window.setTimeout(function () {
                    var child = childrenCollection.children().last();
                    var scrollContainer = child.closest(".BPT-HtmlTree-ScrollContainer")[0];
                    
                    // Don't move horizontally
                    if (scrollContainer) {
                        var x = scrollContainer.scrollLeft;
                        if (toolwindowHelpers.scrollIntoView(child.children(".BPT-HtmlTreeItem-Header")[0], scrollContainer)) {
                            childrenCollection.parent()[0].scrollIntoView(true);
                            scrollContainer.scrollLeft = x;
                        }
                    }
                }, 0);
            }

            return childrenCollection;
        },
        
        addAttribute: function (name, value) {
            /// <summary>
            ///     Adds an attribute to an HtmlTreeItem in the correct position
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object that the attribute was added to
            /// </returns>
            
            var attributesString = "<span class='BPT-HTML-Attribute-Section'> <span class='BPT-HTML-Attribute'>" + name + "</span><span class='BPT-HTML-Operator'>=</span>\"" +
                                   "<span class='BPT-HTML-Value' data-attrName='" + name + "'>" + value + "</span>\"</span>";

            // Add the new attribute after the existing ones
            var existingAttributes = this.find(".BPT-HtmlTreeItem-Header .BPT-HTML:first").children(".BPT-HTML-Attribute-Section");
            if (existingAttributes.length === 0) {
                this.find(".BPT-HtmlTreeItem-Header .BPT-HTML-Tag:first").after($(attributesString));
            } else {
                $(existingAttributes[existingAttributes.length - 1]).after($(attributesString));
            }
            
            return this;
        },
        
        showLoading: function (text) {
            /// <summary>
            ///     Shows a loading message if the item does not already have one
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object
            /// </returns>
            
            if (!this.hasClass("BPT-HtmlTreeItem-ShowingLoader")) {
                var newRowHtml = "<div class='BPT-HtmlTreeItem BPT-HtmlTreeItem-Loading'>" + text + "</div>";
                this.children(".BPT-HtmlTreeItem-Header").append($(newRowHtml));
                this.addClass("BPT-HtmlTreeItem-ShowingLoader");
            }
            return this;
        },
        
        hideLoading: function () {
            /// <summary>
            ///     Removes an existing loading message
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object
            /// </returns>        

            if (this.hasClass("BPT-HtmlTreeItem-ShowingLoader")) {
                this.removeClass("BPT-HtmlTreeItem-ShowingLoader");
                this.children(".BPT-HtmlTreeItem-Header").children(".BPT-HtmlTreeItem-Loading").remove();
            }
            return this;
        },        
        
        getChildren: function () {
            /// <summary>
            ///     Gets a jquery object of the children of this HtmlTreeItem
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object of the children
            /// </returns>
            
            return this.children(".BPT-HtmlTree-ChildCollection").children(".BPT-HtmlTreeItem");
        },
        
        isCollapsed: function () {
            /// <summary>
            ///     Gets the collasped state of the HtmlTreeItem
            /// </summary>  
            /// <returns type="Object">
            ///     True if the HtmlTreeItem is collapsed, false if it is expanded
            /// </returns>
            
            return this.hasClass("BPT-HtmlTreeItem-Collapsed");
        },
        
        isExpanded: function () {
            /// <summary>
            ///     Gets the expanded state of the HtmlTreeItem
            /// </summary>  
            /// <returns type="Object">
            ///     True if the HtmlTreeItem is expanded, false if it is collapsed
            /// </returns>
            
            return this.hasClass("BPT-HtmlTreeItem-Expanded");
        },
        
        changeExpandableState: function (nowExpandable) {
            /// <summary>
            ///     Changes the state of an HtmlTreeItem element to be expandable or not
            /// </summary>  
            /// <param name="nowExpandable" type="Boolean">
            ///     True if the element should now be expandable, False otherwise
            /// </param>              
            /// <returns type="Object">
            ///     The jquery object that was changed
            /// </returns> 
            
            if (nowExpandable) {
                // Now expandable
                var expandIcon = $("<div class='BPT-HtmlTreeItem-ExpandIcon' />");
                this.prepend(expandIcon);
                
                this.find(".BPT-HtmlTreeItem-CollapsedFooter").prepend($("<span>...</span>"));
                this.find(".BPT-HTML-Text").remove();

                // Starts off collapsed if we have children
                this.addClass("BPT-HtmlTreeItem-Collapsed");
            } else {
                // No longer expandable
                this.remove(".BPT-HtmlTreeItem-ExpandIcon");
                this.find(".BPT-HtmlTreeItem-CollapsedFooter").remove(":first-child");
            }
            
            return this;
        },
        
        toggle: function (onExpandComplete) {
            /// <summary>
            ///     Toggles a row between collapsed and expanded views
            /// </summary>
            /// <param name="onExpandComplete" type="Function">
            ///     An optional callback that will be triggered when the toggle expansion has finished
            /// </param>            
            /// <returns type="Object">
            ///     The jquery object that was toggled
            /// </returns> 
        
            // Get the callback
            var toggleCallback = this.data("toggleCallback");
            toggleCallback = (toggleCallback ? toggleCallback : this.parent().data("toggleCallback"));
            
            // If it has a callback then it can be expanded
            if (toggleCallback) {
                // Fire the Begin code marker
                toolwindowHelpers.codeMarker(toolwindowHelpers.codeMarkers.perfBrowserTools_DiagnosticsToolWindowsTreeViewToggleBegin);
            
                if (this.hasClass("BPT-HtmlTreeItem-Collapsed")) {
                    // Expand
                    this.removeClass("BPT-HtmlTreeItem-Collapsed");
                    toggleCallback(true, this, this.attr("data-id"), onExpandComplete);
                    this.addClass("BPT-HtmlTreeItem-Expanded");
                } else {
                
                    if (this.hasClass("BPT-HtmlTreeItem-HiddenRoot")) {
                        return this;
                    }
                
                    // Collapse
                    this.removeClass("BPT-HtmlTreeItem-Expanded");
                    toggleCallback(false, this, this.attr("data-id"));
                    this.children(".BPT-HtmlTree-ChildCollection").remove();
                    this.addClass("BPT-HtmlTreeItem-Collapsed");

                    // Fire the End code marker
                    toolwindowHelpers.codeMarker(toolwindowHelpers.codeMarkers.perfBrowserTools_DiagnosticsToolWindowsTreeViewToggleEnd);
                }
            }
            return this;
        },
        
        getSelected: function () {
            /// <summary>
            ///     Gets the currently selected HtmlTreeItem
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object that is selected
            /// </returns> 
            
            if (this.hasClass("BPT-HtmlTreeItem-Selected")) {
                return this;
            }
            
            var rootElement = this.closest(".BPT-HtmlTree");
            rootElement = (rootElement.length > 0 ? rootElement : this);
            return rootElement.find(".BPT-HtmlTreeItem-Selected:first");
        },
        
        select: function () {
            /// <summary>
            ///     Selects the HtmlTreeItem
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object that was selected
            /// </returns> 
            
            var rootElement = this.closest(".BPT-HtmlTree");
            rootElement.find(".BPT-HtmlTreeItem-Selected").removeClass("BPT-HtmlTreeItem-Selected");

            this.addClass("BPT-HtmlTreeItem-Selected");
            
            var selectCallback = this.data("selectCallback");
            selectCallback = (selectCallback ? selectCallback : this.parent().data("selectCallback"));
            if (selectCallback) {
                selectCallback(this, this.attr("data-id"), this.attr("data-tag"));
            }
            return this;
        },
        
        clear: function () {
            /// <summary>
            ///     Removes all children
            /// </summary>  
            /// <returns type="Object">
            ///     The jquery object that was cleared
            /// </returns> 
            
            // We are removing all our children, so select ourselves instead (the parent node)
            var selectedChild = this.children().find(".BPT-HtmlTreeItem-Selected");
            if (selectedChild.length > 0) {
                methods.select.call(this);
            }
            
            this.children(".BPT-HtmlTree-ChildCollection").remove();
            if (this.hasClass("BPT-HtmlTree-Container")) {
                this.removeAttr("tabindex");
            }
            return this;
        }
    };

    $.fn.htmlTreeView = function (method) {
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof method === "object" || !method) {
            return methods.init.apply(this, arguments);
        }
    };
}(jQuery));


// SIG // Begin signature block
// SIG // MIIajQYJKoZIhvcNAQcCoIIafjCCGnoCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFE4X6c4yMV0C
// SIG // IqCrOXQxSv6G8mYHoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSAMIIEfAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGiMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBQoiDoS2np6JbXuzVTXr0Ce
// SIG // 6c6DLjBCBgorBgEEAYI3AgEMMTQwMqAYgBYAaAB0AG0A
// SIG // bAB0AHIAZQBlAC4AagBzoRaAFGh0dHA6Ly9taWNyb3Nv
// SIG // ZnQuY29tMA0GCSqGSIb3DQEBAQUABIIBAGUNLr7p71Mk
// SIG // /IyqYKIwvhHIEBjkXDiwvnC1/xZ5tC7pebeg1h/QlIpH
// SIG // e+cmYuscC/EXS0/IM3nz3s+xwU5nahjUIIywEeYFukR9
// SIG // LC19hMYYQlmpInp81PXYWhDGCRT9/kn6TaCK9Cl7i8Kx
// SIG // 3Yz8HyAHzvk4ioYiYF2G65VglN3jSTINkwf5iw1qWBR7
// SIG // ACxm/zImhcCSfVUIabp8ioF/IXL6jtxxhZ8Uh5eALd9Q
// SIG // 8jNwZF4zT0q+yJ83VnguRnscWs+2FIeuSbxI+1oM8srP
// SIG // veomZFj32ZI6NRFgURD0n4FaTE7WhD0BbWzxXKHxNBqM
// SIG // qO4PfvkM7GrVeJmYRxlAhjOhggIfMIICGwYJKoZIhvcN
// SIG // AQkGMYICDDCCAggCAQEwgYUwdzELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRpbWUtU3Rh
// SIG // bXAgUENBAgphApJKAAAAAAAgMAkGBSsOAwIaBQCgXTAY
// SIG // BgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3
// SIG // DQEJBTEPFw0xMzAzMTUwNjM0MDRaMCMGCSqGSIb3DQEJ
// SIG // BDEWBBQwmB3SiKQcQLAeMJZOZwZ4a3qvJjANBgkqhkiG
// SIG // 9w0BAQUFAASCAQB0DlO5OPyVLgSADrGAKdIoB5s4GNTH
// SIG // yj16V5jcUVaXt3Wb3fPU/3Qy9q5P+CrOeCqBur+Vlb/8
// SIG // w7hLciECEk6ubLRYps6332z570UHB6bkJYP3lktUB6GG
// SIG // 7MYn5OCxtruS9nMdBrZqC98SwSJlkAdjvoTuCVKtcluM
// SIG // cjHnjncTBa1IB8cl9il5WjqinDkw9okxRhILTsoB5dWt
// SIG // 6sdDt0ZEV5ubsY9XVKxoivxcudEbGm1KHu5czi6POCUJ
// SIG // mGTlKFmlA3vV4pOvLkOyAySa/nZOgyGAEHLHJ8ibF5T+
// SIG // Rap6nPUlt4WQKVlY1ZzEWxkW2PoABOI3mLJ6S9VHQenX
// SIG // ydZQ
// SIG // End signature block
